<?php
// Server-side Anti-Bot Protection
ini_set("display_errors", 0);

// Block known bots from antibots.php list
$__antibotDir = __DIR__;
if (file_exists($__antibotDir . '/antibots.php')) {
    include $__antibotDir . '/antibots.php';
    $__ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if (!empty($bots) && is_array($bots)) {
        foreach ($bots as $bot) {
            if (stripos($__ua, $bot) !== false) {
                http_response_code(403);
                exit('Access denied');
            }
        }
    }
}

// Bot detection and blocking
function detectBot() {
    $botScore = 0;
    $isBot = false;
    
    // Check User Agent
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $suspiciousPatterns = [
        'bot', 'crawler', 'spider', 'scraper', 'curl', 'wget', 'python', 'php',
        'headless', 'phantom', 'selenium', 'webdriver', 'automation', 'test'
    ];
    
    foreach ($suspiciousPatterns as $pattern) {
        if (stripos($userAgent, $pattern) !== false) {
            $botScore += 20;
        }
    }
    
    // Check for missing headers that real browsers send
    $requiredHeaders = [
        'HTTP_ACCEPT',
        'HTTP_ACCEPT_LANGUAGE',
        'HTTP_ACCEPT_ENCODING',
        'HTTP_CONNECTION',
        'HTTP_UPGRADE_INSECURE_REQUESTS'
    ];
    
    foreach ($requiredHeaders as $header) {
        if (!isset($_SERVER[$header])) {
            $botScore += 10;
        }
    }
    
    // Check for suspicious IP patterns
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    if (isCloudProvider($ip)) {
        $botScore += 15;
    }
    
    // Check for missing JavaScript support indicators
    if (!isset($_SERVER['HTTP_ACCEPT']) || strpos($_SERVER['HTTP_ACCEPT'], 'text/html') === false) {
        $botScore += 25;
    }
    
    // Check for automation tool headers
    $automationHeaders = [
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_REAL_IP',
        'HTTP_X_REQUESTED_WITH'
    ];
    
    foreach ($automationHeaders as $header) {
        if (isset($_SERVER[$header])) {
            $botScore += 5;
        }
    }
    
    // Check for missing referrer (bots often don't send referrer)
    if (!isset($_SERVER['HTTP_REFERER'])) {
        $botScore += 10;
    }
    
    // Request timing check removed for high-traffic sites
    // This prevents blocking legitimate users with fast connections
    
    // Check for missing cookies (bots often don't accept cookies)
    if (!isset($_SERVER['HTTP_COOKIE'])) {
        $botScore += 15;
    }
    
    // Request count check removed for high-traffic sites
    // This allows legitimate users to navigate freely
    
    // Determine if bot
    $isBot = $botScore >= 50;
    
    return [
        'isBot' => $isBot,
        'score' => $botScore,
        'userAgent' => $userAgent,
        'ip' => $ip
    ];
}

// Check if IP is from cloud provider
function isCloudProvider($ip) {
    $cloudProviders = [
        'amazonaws.com',
        'googleusercontent.com',
        'microsoft.com',
        'digitalocean.com',
        'linode.com',
        'vultr.com',
        'heroku.com',
        'cloudflare.com'
    ];
    
    // Reverse DNS lookup
    $hostname = gethostbyaddr($ip);
    if ($hostname) {
        foreach ($cloudProviders as $provider) {
            if (strpos($hostname, $provider) !== false) {
                return true;
            }
        }
    }
    
    return false;
}

// Rate limiting removed for high-traffic sites
// This allows multiple legitimate users to access the site simultaneously

// Honeypot field check
function checkHoneypot() {
    // Check for honeypot fields that should be empty
    $honeypotFields = ['website', 'url', 'homepage', 'company'];
    
    foreach ($honeypotFields as $field) {
        if (isset($_POST[$field]) && !empty($_POST[$field])) {
            return false; // Bot filled honeypot
        }
    }
    
    return true;
}

// Main bot detection function
function runServerBotDetection() {
    // Start session if not started
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    // Run all checks
    $botDetection = detectBot();
    $honeypotOk = checkHoneypot();
    
    // If bot detected, log but don't redirect aggressively
    if ($botDetection['isBot'] || !$honeypotOk) {
        // Log the attempt for monitoring
        error_log("Suspicious activity detected: IP={$botDetection['ip']}, Score={$botDetection['score']}, UA={$botDetection['userAgent']}");
        
        // Instead of redirecting, just add a flag for additional monitoring
        $_SESSION['suspicious_activity'] = true;
        $_SESSION['bot_score'] = $botDetection['score'];
    }
    
    return $botDetection;
}

// Run detection on every request
runServerBotDetection();
?>
